/**
 * Storage Adapter for Cross-Browser Extension Compatibility
 *
 * Abstracts differences between Manifest V2 (Firefox) and Manifest V3 (Chrome/Safari)
 * storage APIs to provide a unified interface.
 *
 * MV2: Uses browser.storage.local for all state
 * MV3: Uses chrome.storage.session for ephemeral state (service worker lifecycle)
 *      Uses chrome.storage.local for persistent state
 *
 * Usage:
 *   import StorageAdapter from './adapters/storage-adapter.js';
 *   const storage = new StorageAdapter();
 *   await storage.set('myKey', myValue, { persistent: true });
 *   const value = await storage.get('myKey');
 */

class StorageAdapter {
    constructor() {
        // Detect environment
        this.isServiceWorker = typeof window === 'undefined' && typeof self !== 'undefined';
        this.isManifestV3 = this._detectManifestVersion() === 3;

        // Get the appropriate storage API
        this.browserAPI = typeof browser !== 'undefined' ? browser : chrome;
        this.storageAPI = this.browserAPI?.storage;

        // Cache for frequently accessed values (reduces storage calls)
        this._cache = new Map();
        this._cacheExpiry = new Map();
        this._defaultCacheTTL = 5000; // 5 seconds
    }

    /**
     * Detect manifest version from runtime
     */
    _detectManifestVersion() {
        try {
            const manifest = (typeof browser !== 'undefined' ? browser : chrome)?.runtime?.getManifest();
            return manifest?.manifest_version || 2;
        } catch (e) {
            return 2; // Default to MV2
        }
    }

    /**
     * Get appropriate storage backend based on persistence requirement
     * @param {boolean} persistent - If true, use local storage; if false, use session (MV3) or local (MV2)
     */
    _getStorageBackend(persistent = false) {
        if (!this.storageAPI) {
            console.error('StorageAdapter: No storage API available');
            return null;
        }

        // MV3 with service worker: use session storage for non-persistent data
        if (this.isManifestV3 && !persistent && this.storageAPI.session) {
            return this.storageAPI.session;
        }

        // All other cases: use local storage
        return this.storageAPI.local;
    }

    /**
     * Get value from storage
     * @param {string|string[]} key - Key(s) to retrieve
     * @param {object} options - { persistent: boolean, useCache: boolean }
     * @returns {Promise<any>} - Retrieved value or undefined
     */
    async get(key, options = {}) {
        const { persistent = false, useCache = true } = options;

        // Check cache first
        if (useCache && typeof key === 'string') {
            const cached = this._getFromCache(key);
            if (cached !== undefined) {
                return cached;
            }
        }

        const storage = this._getStorageBackend(persistent);
        if (!storage) {
            return undefined;
        }

        try {
            const result = await storage.get(key);

            // Handle single key vs array of keys
            if (typeof key === 'string') {
                const value = result[key];
                if (useCache) {
                    this._setCache(key, value);
                }
                return value;
            }

            return result;
        } catch (error) {
            console.error('StorageAdapter.get error:', error);
            return undefined;
        }
    }

    /**
     * Set value in storage
     * @param {string|object} key - Key or object of key-value pairs
     * @param {any} value - Value to store (ignored if key is object)
     * @param {object} options - { persistent: boolean }
     * @returns {Promise<boolean>} - Success status
     */
    async set(key, value, options = {}) {
        const { persistent = false } = options;

        const storage = this._getStorageBackend(persistent);
        if (!storage) {
            return false;
        }

        try {
            const data = typeof key === 'object' ? key : { [key]: value };
            await storage.set(data);

            // Update cache
            Object.entries(data).forEach(([k, v]) => {
                this._setCache(k, v);
            });

            return true;
        } catch (error) {
            console.error('StorageAdapter.set error:', error);
            return false;
        }
    }

    /**
     * Remove value from storage
     * @param {string|string[]} key - Key(s) to remove
     * @param {object} options - { persistent: boolean }
     * @returns {Promise<boolean>} - Success status
     */
    async remove(key, options = {}) {
        const { persistent = false } = options;

        const storage = this._getStorageBackend(persistent);
        if (!storage) {
            return false;
        }

        try {
            await storage.remove(key);

            // Clear from cache
            const keys = Array.isArray(key) ? key : [key];
            keys.forEach(k => this._clearCache(k));

            return true;
        } catch (error) {
            console.error('StorageAdapter.remove error:', error);
            return false;
        }
    }

    /**
     * Clear all storage (use with caution)
     * @param {object} options - { persistent: boolean, both: boolean }
     */
    async clear(options = {}) {
        const { persistent = false, both = false } = options;

        try {
            if (both && this.storageAPI) {
                // Clear both session and local
                if (this.storageAPI.session) {
                    await this.storageAPI.session.clear();
                }
                await this.storageAPI.local.clear();
            } else {
                const storage = this._getStorageBackend(persistent);
                if (storage) {
                    await storage.clear();
                }
            }

            // Clear cache
            this._cache.clear();
            this._cacheExpiry.clear();

            return true;
        } catch (error) {
            console.error('StorageAdapter.clear error:', error);
            return false;
        }
    }

    /**
     * Get all keys in storage
     * @param {object} options - { persistent: boolean }
     * @returns {Promise<string[]>} - Array of keys
     */
    async keys(options = {}) {
        const { persistent = false } = options;

        const storage = this._getStorageBackend(persistent);
        if (!storage) {
            return [];
        }

        try {
            const result = await storage.get(null);
            return Object.keys(result);
        } catch (error) {
            console.error('StorageAdapter.keys error:', error);
            return [];
        }
    }

    // ========== Cache Management ==========

    _getFromCache(key) {
        const expiry = this._cacheExpiry.get(key);
        if (expiry && Date.now() > expiry) {
            this._clearCache(key);
            return undefined;
        }
        return this._cache.get(key);
    }

    _setCache(key, value, ttl = this._defaultCacheTTL) {
        this._cache.set(key, value);
        this._cacheExpiry.set(key, Date.now() + ttl);
    }

    _clearCache(key) {
        this._cache.delete(key);
        this._cacheExpiry.delete(key);
    }

    /**
     * Invalidate all cached values
     */
    invalidateCache() {
        this._cache.clear();
        this._cacheExpiry.clear();
    }

    // ========== MV3 Service Worker Helpers ==========

    /**
     * Restore state after service worker wakes up
     * Call this at the start of message handlers in MV3
     * @param {string[]} keys - Keys to preload into cache
     */
    async rehydrateState(keys) {
        if (!this.isManifestV3) return;

        try {
            const persistent = await this.storageAPI.local.get(keys);
            const session = this.storageAPI.session
                ? await this.storageAPI.session.get(keys)
                : {};

            // Merge session over persistent (session is more recent)
            const merged = { ...persistent, ...session };

            // Populate cache
            Object.entries(merged).forEach(([key, value]) => {
                this._setCache(key, value, 30000); // 30 second TTL for rehydrated state
            });

            return merged;
        } catch (error) {
            console.error('StorageAdapter.rehydrateState error:', error);
            return {};
        }
    }

    /**
     * Persist critical state before service worker terminates
     * @param {object} state - State to persist
     */
    async persistState(state) {
        if (!this.isManifestV3) return;

        try {
            // Store in both session (for quick access) and local (for persistence)
            if (this.storageAPI.session) {
                await this.storageAPI.session.set(state);
            }
            await this.storageAPI.local.set(state);
            return true;
        } catch (error) {
            console.error('StorageAdapter.persistState error:', error);
            return false;
        }
    }
}

// Export singleton instance
const storageAdapter = new StorageAdapter();

// Make available globally for non-module contexts
if (typeof window !== 'undefined') {
    window.StorageAdapter = StorageAdapter;
    window.storageAdapter = storageAdapter;
}

// ES module export
if (typeof module !== 'undefined' && module.exports) {
    module.exports = { StorageAdapter, storageAdapter };
}
