/**
 * OAuth Authentication Module for CarbonGuru Extension
 * Handles WordPress OAuth flow and JWT token management
 */

class OAuthManager {
    constructor() {
        this.clientId = null;
        this.redirectUri = null;
        this.authorizeUrl = null;
        this.jwtToken = null;
        this.tokenExpiry = null;
        this.backendUrl = null;
        this.pendingState = null; // Store state for CSRF validation

        void 0;
    }

    /**
     * Initialize OAuth manager with configuration
     */
    async init(environmentConfig) {
        this.backendUrl = environmentConfig.backendUrl;

        // CarbonGuru WordPress OAuth client identifier (public, not the Google OAuth client_id)
        this.clientId = 'carbonguru_extension_4af6791c6fee9be5';
        this.redirectUri = `${this.backendUrl}/extension/oauth/callback`;
        this.authorizeUrl = `${this.backendUrl}/oauth/authorize`;

        void 0;

        // Load stored JWT token if exists
        const tokenLoaded = await this.loadStoredToken();

        if (tokenLoaded) {
            void 0;
        } else {
            void 0;
        }

        return tokenLoaded;
    }

    /**
     * Load JWT token from storage
     */
    async loadStoredToken() {
        try {
            const storage = typeof browser !== 'undefined' ? browser.storage : chrome.storage;
            const result = await storage.local.get(['jwt_token', 'token_expiry']);

            void 0;

            if (result.jwt_token && result.token_expiry) {
                const now = Date.now();
                const expiry = parseInt(result.token_expiry);
                const expiryDate = new Date(expiry);
                const timeUntilExpiry = (expiry - now) / 1000 / 60; // minutes

                void 0;

                // Check if token is still valid (not expired)
                if (now < expiry) {
                    this.jwtToken = result.jwt_token;
                    this.tokenExpiry = expiry;
                    void 0;
                    void 0;
                    return true;
                } else {
                    void 0;
                    await this.clearToken();
                }
            } else {
                void 0;
            }
        } catch (error) {
            console.error('CarbonGuru OAuth: ❌ Error loading stored token:', error);
        }
        return false;
    }

    /**
     * Save JWT token to storage
     */
    async saveToken(token, expiresIn = 86400) {
        try {
            const storage = typeof browser !== 'undefined' ? browser.storage : chrome.storage;
            const expiry = Date.now() + (expiresIn * 1000); // expiresIn is in seconds
            const expiryDate = new Date(expiry);

            this.jwtToken = token;
            this.tokenExpiry = expiry;

            void 0;
            await storage.local.set({
                'jwt_token': token,
                'token_expiry': expiry
            });

            void 0;
            void 0;
            void 0;
            return true;
        } catch (error) {
            console.error('CarbonGuru OAuth: ❌ Error saving token:', error);
            return false;
        }
    }

    /**
     * Clear stored JWT token
     */
    async clearToken() {
        try {
            const storage = typeof browser !== 'undefined' ? browser.storage : chrome.storage;
            await storage.local.remove(['jwt_token', 'token_expiry']);

            this.jwtToken = null;
            this.tokenExpiry = null;

            void 0;
            return true;
        } catch (error) {
            console.error('CarbonGuru OAuth: Error clearing token:', error);
            return false;
        }
    }

    /**
     * Check if user is authenticated (synchronous - uses cached values)
     * For service worker resilience, use ensureAuthenticated() instead
     */
    isAuthenticated() {
        const hasToken = !!this.jwtToken;
        const hasExpiry = !!this.tokenExpiry;

        void 0;

        if (!hasToken || !hasExpiry) {
            void 0;
            return false;
        }

        // Check if token is expired
        const now = Date.now();
        const isExpired = now >= this.tokenExpiry;
        const minutesUntilExpiry = (this.tokenExpiry - now) / 1000 / 60;

        void 0;

        if (isExpired) {
            void 0;
            this.clearToken();
            return false;
        }

        void 0;
        return true;
    }

    /**
     * Ensure authenticated with service worker resilience
     * Reloads token from storage if in-memory cache is empty (handles service worker suspension)
     * @returns {Promise<boolean>} true if authenticated, false otherwise
     */
    async ensureAuthenticated() {
        // If we have token in memory, use synchronous check
        if (this.jwtToken && this.tokenExpiry) {
            return this.isAuthenticated();
        }

        // In-memory cache is empty - try to reload from storage
        // This handles service worker suspension/resume scenarios
        void 0;
        const loaded = await this.loadStoredToken();

        if (loaded) {
            void 0;
            return this.isAuthenticated();
        }

        void 0;
        return false;
    }

    /**
     * Get authorization header for API requests (async version with storage reload)
     * @returns {Promise<Object|null>} Authorization header object or null if not authenticated
     */
    async getAuthHeaderAsync() {
        const isAuth = await this.ensureAuthenticated();
        if (!isAuth) {
            return null;
        }

        return {
            'Authorization': `Bearer ${this.jwtToken}`
        };
    }

    /**
     * Get authorization header for API requests
     */
    getAuthHeader() {
        if (!this.isAuthenticated()) {
            return null;
        }

        return {
            'Authorization': `Bearer ${this.jwtToken}`
        };
    }

    /**
     * Initiate OAuth flow - opens WordPress authorization page
     */
    async initiateOAuthFlow() {
        try {
            void 0;

            // Generate and store state for CSRF protection
            this.pendingState = this.generateState();

            // Build authorization URL with parameters
            const authUrl = new URL(this.authorizeUrl);
            authUrl.searchParams.append('client_id', this.clientId);
            authUrl.searchParams.append('redirect_uri', this.redirectUri);
            authUrl.searchParams.append('response_type', 'code');
            authUrl.searchParams.append('scope', 'basic email profile');
            authUrl.searchParams.append('state', this.pendingState);

            void 0;

            // Open authorization page in new tab
            const tabsAPI = typeof browser !== 'undefined' ? browser.tabs : chrome.tabs;
            const tab = await tabsAPI.create({
                url: authUrl.toString(),
                active: true
            });

            void 0;

            return {
                success: true,
                tabId: tab.id,
                message: 'Please authorize in the opened tab'
            };

        } catch (error) {
            console.error('CarbonGuru OAuth: Error initiating OAuth flow:', error);
            return {
                success: false,
                error: error.message
            };
        }
    }

    /**
     * Handle OAuth callback with authorization code
     */
    async handleCallback(code, state) {
        try {
            void 0;
            void 0;

            // Validate state parameter to prevent CSRF attacks
            if (!state || !this.pendingState || state !== this.pendingState) {
                console.error('CarbonGuru OAuth: ❌ State validation failed - possible CSRF attack');
                this.pendingState = null; // Clear pending state
                throw new Error('OAuth state validation failed - request may have been tampered with');
            }
            void 0;
            this.pendingState = null; // Clear pending state after validation

            // Exchange authorization code for JWT token
            const tokenExchangeUrl = `${this.backendUrl}/api/auth/wordpress-oauth`;
            void 0;

            const response = await fetch(tokenExchangeUrl, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({
                    code: code,
                    redirect_uri: this.redirectUri
                })
            });

            void 0;

            if (!response.ok) {
                const errorText = await response.text();
                console.error('CarbonGuru OAuth: ❌ Token exchange failed:', errorText);
                throw new Error(`Token exchange failed: ${response.status} ${errorText}`);
            }

            const data = await response.json();
            void 0;

            if (!data.token) {
                throw new Error('No token received from backend');
            }

            // Save JWT token
            const expiresIn = data.expires_in || 86400; // 24 hours default
            const tokenSaved = await this.saveToken(data.token, expiresIn);

            if (!tokenSaved) {
                throw new Error('Failed to save token to storage');
            }

            // Save user info to storage
            if (data.user) {
                const storage = typeof browser !== 'undefined' ? browser.storage : chrome.storage;
                await storage.local.set({ 'user_info': data.user });
                void 0;
            }

            void 0;

            // Notify all content scripts that authentication status changed
            this.notifyAuthenticationChange(true, data.user);

            return {
                success: true,
                user: data.user,
                message: 'Authentication successful'
            };

        } catch (error) {
            console.error('CarbonGuru OAuth: ❌ Callback error:', error);
            return {
                success: false,
                error: error.message
            };
        }
    }

    /**
     * Generate random state for OAuth CSRF protection
     */
    generateState() {
        const array = new Uint8Array(16);
        crypto.getRandomValues(array);
        return Array.from(array, byte => byte.toString(16).padStart(2, '0')).join('');
    }

    /**
     * Logout - clear JWT token and user info
     */
    async logout() {
        void 0;
        await this.clearToken();

        // Clear user info
        try {
            const storage = typeof browser !== 'undefined' ? browser.storage : chrome.storage;
            await storage.local.remove(['user_info']);
            void 0;
        } catch (error) {
            console.error('CarbonGuru OAuth: Error clearing user info:', error);
        }

        // Notify all content scripts that authentication status changed
        this.notifyAuthenticationChange(false, null);

        return {
            success: true,
            message: 'Logged out successfully'
        };
    }

    /**
     * Notify all content scripts that authentication status changed
     * This allows content scripts to update their UI without requiring page reload
     */
    async notifyAuthenticationChange(isAuthenticated, user) {
        void 0;

        const tabsAPI = typeof browser !== 'undefined' ? browser.tabs : chrome.tabs;

        try {
            // Query all tabs asynchronously
            const tabs = await tabsAPI.query({});
            void 0;

            let successCount = 0;
            let failureCount = 0;

            // Process tabs sequentially to avoid overwhelming the message system
            for (const tab of tabs) {
                // Skip extension internal pages and special URLs
                if (!tab.url || tab.url.startsWith('about:') ||
                    tab.url.startsWith('chrome:') ||
                    tab.url.startsWith('moz-extension:') ||
                    tab.url.startsWith('edge:') ||
                    tab.url.startsWith('safari:')) {
                    void 0;
                    continue;
                }

                try {
                    // First check if content script is ready with a ping
                    await this.sendMessageWithTimeout(tab.id, { action: 'ping' }, 1000);

                    // Content script responded, send auth change notification
                    await this.sendMessageWithTimeout(tab.id, {
                        action: 'authenticationStatusChanged',
                        isAuthenticated: isAuthenticated,
                        user: user
                    }, 3000);

                    successCount++;
                    void 0;

                } catch (error) {
                    failureCount++;
                    void 0;
                }
            }

            void 0;

        } catch (error) {
            console.error('CarbonGuru OAuth: Error querying tabs:', error);
        }
    }

    /**
     * Send message to tab with timeout
     * Returns promise that rejects if tab doesn't respond within timeout
     */
    sendMessageWithTimeout(tabId, message, timeout = 5000) {
        return new Promise((resolve, reject) => {
            const timer = setTimeout(() => {
                reject(new Error(`Timeout after ${timeout}ms`));
            }, timeout);

            const tabsAPI = typeof browser !== 'undefined' ? browser.tabs : chrome.tabs;

            tabsAPI.sendMessage(tabId, message).then((response) => {
                clearTimeout(timer);
                resolve(response);
            }).catch((error) => {
                clearTimeout(timer);
                reject(error);
            });
        });
    }

    /**
     * Send message to tab with retry logic
     */
    async sendMessageWithRetry(tabId, message, maxRetries = 3) {
        const tabsAPI = typeof browser !== 'undefined' ? browser.tabs : chrome.tabs;
        let lastError = null;

        for (let attempt = 1; attempt <= maxRetries; attempt++) {
            try {
                // Small delay between retries
                if (attempt > 1) {
                    await new Promise(resolve => setTimeout(resolve, 200 * attempt));
                    void 0;
                }

                const response = await tabsAPI.sendMessage(tabId, message);
                void 0;
                return response; // Success
            } catch (error) {
                lastError = error;
                void 0;
            }
        }

        throw lastError; // All retries failed
    }
}

// Export for use in background script
if (typeof module !== 'undefined' && module.exports) {
    module.exports = OAuthManager;
}
