/**
 * Gmail Selection Detector
 * Detects checked/selected emails in Gmail inbox using ARIA attributes
 *
 * VALIDATED: 2025-11-06 with real Gmail data (4/4 thread IDs extracted successfully)
 * Uses data-thread-id attributes on span elements (not href links)
 */

class GmailSelectionDetector {
    constructor() {
        this.observer = null;
        this.selectedCount = 0;
        this.selectedRows = [];
        this.listeners = new Map();
        this.mainArea = null;
    }

    /**
     * Initialize the detector - find main area and start watching
     * @returns {boolean} True if initialization successful
     */
    initialize() {
        this.mainArea = this.findMainArea();
        if (!this.mainArea) {
            console.error('[GmailSelectionDetector] Could not find main area');
            return false;
        }

        this.startWatching();
        this.updateSelection();

        void 0;
        return true;
    }

    /**
     * Find the main mail area to scope our checkbox searches
     * @returns {HTMLElement|null} Main mail area element
     */
    findMainArea() {
        // Try multiple selectors for robustness
        return document.querySelector('div[role="main"]') ||
               document.querySelector('.AO') ||  // Gmail inbox container
               document.body;
    }

    /**
     * Get all checked checkboxes using ARIA attributes
     * @returns {HTMLElement[]} Array of checked checkbox elements
     */
    getCheckedBoxes() {
        if (!this.mainArea) return [];
        return Array.from(
            this.mainArea.querySelectorAll('div[role="checkbox"][aria-checked="true"]')
        );
    }

    /**
     * Extract message IDs from selected rows
     * @returns {string[]} Array of Gmail thread IDs
     */
    getMessageIdsFromSelection() {
        const checkedBoxes = this.getCheckedBoxes();
        const messageIds = [];

        for (const checkbox of checkedBoxes) {
            const row = this.findRowFromCheckbox(checkbox);
            if (row) {
                const messageId = this.extractMessageId(row);
                if (messageId) {
                    messageIds.push(messageId);
                }
            }
        }

        void 0;
        return messageIds;
    }

    /**
     * Walk up from checkbox to find the row container
     * @param {HTMLElement} checkbox Checkbox element
     * @returns {HTMLElement|null} Row container element
     */
    findRowFromCheckbox(checkbox) {
        // Try multiple container patterns
        return checkbox.closest('[role="row"]') ||
               checkbox.closest('.zA') ||  // Gmail row class
               checkbox.closest('tr') ||
               checkbox.closest('div[data-message-id]');
    }

    /**
     * Extract message ID from row element
     *
     * VALIDATED METHOD (2025-11-06): Gmail stores thread IDs in data-thread-id
     * attributes on span elements, NOT in href links as initially assumed.
     *
     * Real-world validation confirmed this extraction method works with actual
     * Gmail inbox structure. Tested with 4 checked emails, all IDs extracted successfully.
     *
     * @param {HTMLElement} row Row container element
     * @returns {string|null} Gmail thread ID or null if not found
     */
    extractMessageId(row) {
        // VALIDATED: Method 1 - data-thread-id attribute (PRIMARY)
        // Format: "#thread-f:1836995587227269341" (19-digit decimal thread ID)
        const spanWithThreadId = row.querySelector('[data-thread-id]');
        if (spanWithThreadId) {
            const fullId = spanWithThreadId.getAttribute('data-thread-id');
            // Remove "#thread-f:" prefix to get numeric thread ID
            const threadId = fullId.replace('#thread-f:', '');
            return threadId;
        }

        // VALIDATED: Method 2 - data-legacy-thread-id attribute (FALLBACK)
        // Format: "197e51c0f6a438dd" (16-character hex thread ID)
        const spanWithLegacy = row.querySelector('[data-legacy-thread-id]');
        if (spanWithLegacy) {
            return spanWithLegacy.getAttribute('data-legacy-thread-id');
        }

        // These methods below are legacy/unvalidated and may not work in current Gmail:

        // Method 3: data-message-id attribute (untested)
        const dataAttr = row.getAttribute('data-message-id');
        if (dataAttr) return dataAttr;

        // Method 4: ID attribute pattern (untested)
        const rowId = row.getAttribute('id');
        if (rowId && rowId.startsWith(':r')) return rowId;

        void 0;
        return null;
    }

    /**
     * Start MutationObserver to watch for selection changes
     */
    startWatching() {
        this.observer = new MutationObserver((mutations) => {
            for (const mutation of mutations) {
                if (mutation.type === 'attributes' &&
                    mutation.attributeName === 'aria-checked' &&
                    mutation.target.getAttribute('role') === 'checkbox') {
                    this.updateSelection();
                    break;
                }
            }
        });

        this.observer.observe(this.mainArea, {
            subtree: true,
            attributes: true,
            attributeFilter: ['aria-checked']
        });

        void 0;
    }

    /**
     * Stop watching for changes
     */
    stopWatching() {
        if (this.observer) {
            this.observer.disconnect();
            this.observer = null;
            void 0;
        }
    }

    /**
     * Update current selection state and notify listeners
     */
    updateSelection() {
        const checkedBoxes = this.getCheckedBoxes();
        const newCount = checkedBoxes.length;

        // Only emit if count changed
        if (newCount !== this.selectedCount) {
            const oldCount = this.selectedCount;
            this.selectedCount = newCount;
            this.selectedRows = checkedBoxes.map(cb => this.findRowFromCheckbox(cb)).filter(Boolean);

            void 0;
            this.emit('selection-changed', this.selectedCount);
        }
    }

    /**
     * Get current selection count
     * @returns {number} Number of selected emails
     */
    getSelectionCount() {
        return this.selectedCount;
    }

    /**
     * Event system - register listener
     * @param {string} event Event name
     * @param {Function} callback Callback function
     */
    on(event, callback) {
        if (!this.listeners.has(event)) {
            this.listeners.set(event, []);
        }
        this.listeners.get(event).push(callback);
    }

    /**
     * Event system - remove listener
     * @param {string} event Event name
     * @param {Function} callback Callback function to remove
     */
    off(event, callback) {
        const callbacks = this.listeners.get(event);
        if (callbacks) {
            const index = callbacks.indexOf(callback);
            if (index !== -1) {
                callbacks.splice(index, 1);
            }
        }
    }

    /**
     * Event system - emit event
     * @param {string} event Event name
     * @param {...any} args Arguments to pass to callbacks
     */
    emit(event, ...args) {
        const callbacks = this.listeners.get(event);
        if (callbacks) {
            callbacks.forEach(cb => cb(...args));
        }
    }

    /**
     * Cleanup
     */
    destroy() {
        this.stopWatching();
        this.listeners.clear();
        this.selectedCount = 0;
        this.selectedRows = [];
        void 0;
    }
}

// Export for use in other modules
if (typeof module !== 'undefined' && module.exports) {
    module.exports = GmailSelectionDetector;
}
