/**
 * Gmail API Integration Module
 * Handles Gmail API communication and email processing for CarbonGuru extension
 */

class GmailIntegration {
    constructor(backendUrl, getAuthHeaderFn) {
        this.backendUrl = backendUrl;
        this.getAuthHeader = getAuthHeaderFn; // Function to get JWT auth header
        this.processingQueue = [];
        this.isProcessing = false;
    }

    /**
     * Process a single email by message ID
     * @param {string} messageId - Gmail message ID
     * @param {function} statusCallback - Callback for status updates
     * @returns {Promise<Object>} Processing result
     */
    async processSingleEmail(messageId, statusCallback = null) {
        try {
            void 0;

            if (statusCallback) {
                statusCallback({
                    type: 'info',
                    message: 'Initiating email processing...',
                    stage: 'init'
                });
            }

            // Call backend API to initiate async email processing
            // Backend will use stored OAuth tokens to fetch from Gmail API
            const authHeader = await this.getAuthHeader();
            if (!authHeader) {
                throw new Error('Not authenticated. Please log in to CarbonGuru.');
            }

            const response = await fetch(`${this.backendUrl}/api/gmail/process-emails`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    ...authHeader
                },
                body: JSON.stringify({
                    message_id: messageId, // Process specific message
                    single_message: true
                })
            });

            if (!response.ok) {
                const errorData = await response.json();

                // Handle Gmail not connected error
                if (response.status === 401 && errorData.error === 'Gmail not connected') {
                    throw new Error('Gmail not connected. Please connect your Gmail account in CarbonGuru settings.');
                }

                throw new Error(errorData.message || `HTTP ${response.status}: ${response.statusText}`);
            }

            const result = await response.json();
            void 0;

            if (!result.success) {
                console.error('CarbonGuru Gmail: Backend returned error:', result);
                throw new Error(result.message || result.error || 'Email processing failed');
            }

            // Backend returns process_id for async tracking
            const processId = result.process_id;
            void 0;

            if (statusCallback) {
                statusCallback({
                    type: 'info',
                    message: 'Processing email...',
                    stage: 'processing'
                });
            }

            // Poll for processing status updates
            const finalResult = await this.pollEmailProcessing(processId, statusCallback);

            return finalResult;

        } catch (error) {
            console.error('CarbonGuru Gmail: Error processing single email:', error);

            if (statusCallback) {
                statusCallback({
                    type: 'error',
                    message: error.message,
                    stage: 'error'
                });
            }

            return {
                success: false,
                error: error.message
            };
        }
    }

    /**
     * Poll backend for email processing status updates
     * @param {string} processId - Process ID for async tracking (e.g., "gmail_1762188238_5365")
     * @param {function} statusCallback - Callback for status updates
     * @returns {Promise<Object>} Final processing result
     */
    async pollEmailProcessing(processId, statusCallback = null) {
        void 0;
        const maxAttempts = 120; // 2 minutes (120 * 1 second)
        let attempts = 0;
        let lastActivityCount = 0;

        const poll = async () => {
            attempts++;
            void 0;

            try {
                const authHeader = await this.getAuthHeader();
                void 0;
                const response = await fetch(`${this.backendUrl}/api/activity-status?session_id=${processId}`, {
                    method: 'GET',
                    headers: {
                        ...authHeader
                    }
                });

                void 0;
                if (!response.ok) {
                    throw new Error(`Polling failed: ${response.status}`);
                }

                const statusData = await response.json();
                void 0;

                // Send progress updates via statusCallback (structured messages from /api/activity-status)
                if (statusCallback && statusData.status_messages && statusData.status_messages.length > 0) {
                    // Check for new status messages (reversed because API returns newest first)
                    const messages = statusData.status_messages.reverse();
                    if (messages.length > lastActivityCount) {
                        const newMessages = messages.slice(lastActivityCount);
                        lastActivityCount = messages.length;

                        // Send each new message
                        for (const activity of newMessages) {
                            statusCallback({
                                type: 'info',
                                message: activity.message || 'Processing...',
                                stage: activity.content_type || 'processing',
                                activity: activity
                            });
                        }
                    }
                }

                // Check if processing is complete by polling AsyncResult status
                // We need to also check /api/gmail/status for completion
                const asyncAuthHeader = await this.getAuthHeader();
                const asyncResponse = await fetch(`${this.backendUrl}/api/gmail/status/${processId}`, {
                    method: 'GET',
                    headers: { ...asyncAuthHeader }
                });

                if (asyncResponse.ok) {
                    const asyncData = await asyncResponse.json();

                    // Check if processing is complete
                    if (asyncData.status === 'completed') {
                        void 0;

                        if (statusCallback) {
                            statusCallback({
                                type: 'success',
                                message: 'Email processing complete',
                                stage: 'complete'
                            });
                        }

                        // Return final result
                        return {
                            success: true,
                            document: asyncData.result && asyncData.result.documents && asyncData.result.documents.length > 0
                                ? asyncData.result.documents[0]
                                : null,
                            items_extracted: asyncData.result ? asyncData.result.items_extracted : 0,
                            total_carbon: asyncData.result ? (asyncData.result.total_carbon_footprint || 0) : 0,
                            total_cost: asyncData.result ? (asyncData.result.total_carbon_cost || 0) : 0
                        };
                    }

                    // Check if processing failed
                    if (asyncData.status === 'error' || asyncData.status === 'failed') {
                        throw new Error(asyncData.error || 'Email processing failed');
                    }
                }

                // Still processing - continue polling
                if (attempts < maxAttempts) {
                    await new Promise(resolve => setTimeout(resolve, 1000)); // Poll every 1 second
                    return poll();
                } else {
                    throw new Error('Email processing timeout - please try again');
                }

            } catch (error) {
                console.error('CarbonGuru Gmail: Polling error:', error);

                if (statusCallback) {
                    statusCallback({
                        type: 'error',
                        message: error.message,
                        stage: 'error'
                    });
                }

                throw error;
            }
        };

        return poll();
    }

    /**
     * Process multiple emails in batch
     * @param {Array<string>} messageIds - Array of Gmail message IDs
     * @param {function} statusCallback - Callback for status updates (called for each email)
     * @returns {Promise<Object>} Batch processing result
     */
    async processBatchEmails(messageIds, statusCallback = null) {
        try {
            void 0;

            const authHeader = await this.getAuthHeader();
            if (!authHeader) {
                throw new Error('Not authenticated. Please log in to CarbonGuru.');
            }

            // Process emails sequentially (most recent → oldest)
            const results = {
                success: true,
                total_emails: messageIds.length,
                processed_emails: 0,
                failed_emails: 0,
                total_items: 0,
                total_carbon: 0,
                total_cost: 0,
                documents: []
            };

            for (let i = 0; i < messageIds.length; i++) {
                const messageId = messageIds[i];

                try {
                    void 0;

                    if (statusCallback) {
                        statusCallback({
                            type: 'info',
                            message: `Processing email ${i + 1} of ${messageIds.length}...`,
                            stage: 'batch_processing',
                            current: i + 1,
                            total: messageIds.length
                        });
                    }

                    // Process individual email
                    const emailResult = await this.processSingleEmail(messageId, statusCallback);

                    if (emailResult.success && emailResult.document) {
                        results.processed_emails++;
                        results.total_items += emailResult.items_extracted || 0;
                        results.total_carbon += emailResult.total_carbon || 0;
                        results.total_cost += emailResult.total_cost || 0;
                        results.documents.push(emailResult.document);
                    } else {
                        results.failed_emails++;
                        void 0;
                    }

                    // Small delay between emails to avoid rate limiting
                    if (i < messageIds.length - 1) {
                        await new Promise(resolve => setTimeout(resolve, 500));
                    }

                } catch (emailError) {
                    console.error(`CarbonGuru Gmail: Error processing email ${i + 1}:`, emailError);
                    results.failed_emails++;
                }
            }

            void 0;

            return results;

        } catch (error) {
            console.error('CarbonGuru Gmail: Batch processing error:', error);

            if (statusCallback) {
                statusCallback({
                    type: 'error',
                    message: error.message,
                    stage: 'error'
                });
            }

            return {
                success: false,
                error: error.message
            };
        }
    }

    /**
     * Process emails from date range
     * @param {Object} options - Processing options
     * @param {number} options.days_back - Number of days to look back (default: 30)
     * @param {Array<string>} options.document_types - Filter by document types
     * @param {function} statusCallback - Callback for status updates
     * @returns {Promise<Object>} Processing result
     */
    async processEmailsByDateRange(options = {}, statusCallback = null) {
        try{
            const days_back = options.days_back || 30;
            const limit = options.limit || null;
            const document_types = options.document_types || null;
            const category = options.category || null;

            const categoryMsg = category ? ` (category: ${category})` : '';
            const limitMsg = limit ? ` (limit ${limit})` : '';
            void 0;

            if (statusCallback) {
                const message = category ?
                    (limit ? `Fetching ${limit} ${category} emails...` : `Fetching ${category} emails...`) :
                    (limit ? `Fetching ${limit} most recent emails...` : `Fetching emails from last ${days_back} days...`);
                statusCallback({
                    type: 'info',
                    message: message,
                    stage: 'fetch'
                });
            }

            const authHeader = await this.getAuthHeader();
            if (!authHeader) {
                throw new Error('Not authenticated. Please log in to CarbonGuru.');
            }

            // Build request body
            const requestBody = {
                days_back: days_back
            };
            if (limit) requestBody.limit = limit;
            if (document_types) requestBody.document_types = document_types;
            if (category) requestBody.category = category;

            // Call backend API with date range
            const response = await fetch(`${this.backendUrl}/api/gmail/process-emails`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    ...authHeader
                },
                body: JSON.stringify(requestBody)
            });

            if (!response.ok) {
                const errorData = await response.json();

                if (response.status === 401 && errorData.error === 'Gmail not connected') {
                    throw new Error('Gmail not connected. Please connect your Gmail account in CarbonGuru settings.');
                }

                throw new Error(errorData.message || `HTTP ${response.status}: ${response.statusText}`);
            }

            const result = await response.json();

            if (!result.success) {
                // Check for user input required (geography selection)
                if (result.user_input_required) {
                    return {
                        success: false,
                        user_input_required: true,
                        input_type: result.input_type,
                        message: result.message,
                        data: result.data
                    };
                }

                throw new Error(result.message || 'Email processing failed');
            }

            // Backend returns process_id for async tracking (same as processSingleEmail)
            const processId = result.process_id;
            void 0;

            if (statusCallback) {
                statusCallback({
                    type: 'info',
                    message: 'Processing emails...',
                    stage: 'processing'
                });
            }

            // Poll for processing status updates (same pattern as processSingleEmail)
            const finalResult = await this.pollEmailProcessing(processId, statusCallback);

            return finalResult;

        } catch (error) {
            console.error('CarbonGuru Gmail: Date range processing error:', error);

            if (statusCallback) {
                statusCallback({
                    type: 'error',
                    message: error.message,
                    stage: 'error'
                });
            }

            return {
                success: false,
                error: error.message
            };
        }
    }

    /**
     * Check Gmail connection status
     * @returns {Promise<Object>} Connection status
     */
    async checkGmailConnection() {
        try {
            const authHeader = await this.getAuthHeader();
            if (!authHeader) {
                console.error('CarbonGuru Gmail: No auth header available');
                return {
                    connected: false,
                    error: 'Not authenticated'
                };
            }

            void 0;

            const response = await fetch(`${this.backendUrl}/api/gmail/status`, {
                method: 'GET',
                headers: {
                    ...authHeader
                }
            });

            void 0;

            if (!response.ok) {
                console.error('CarbonGuru Gmail: Status check failed with HTTP', response.status);
                return {
                    connected: false,
                    error: `HTTP ${response.status}`
                };
            }

            const result = await response.json();
            void 0;

            return {
                connected: result.connected || false,
                email: result.email || null,
                error: result.error || null
            };

        } catch (error) {
            console.error('CarbonGuru Gmail: Connection check error:', error);
            return {
                connected: false,
                error: error.message
            };
        }
    }
}

// Export for use in background script
if (typeof module !== 'undefined' && module.exports) {
    module.exports = GmailIntegration;
}
