/**
 * Floating Tile Gmail Extensions
 * Adds Gmail-specific UI for batch email processing and scope selection
 */

class FloatingTileGmailExtension {
    constructor(contentScript) {
        this.contentScript = contentScript;
        this.gmailDetector = null;
        this.gmailIntegration = null;
        this.scopeSelectorVisible = false;
        this.batchProcessingActive = false;
        this.selectionDetector = null;  // NEW: For batch selection feature
    }

    /**
     * Initialize Gmail extensions
     */
    async init(gmailDetector, gmailIntegration) {
        this.gmailDetector = gmailDetector;
        this.gmailIntegration = gmailIntegration;

        // NEW: Initialize Gmail selection detector for batch selection feature
        this.selectionDetector = new GmailSelectionDetector();
        const detectorReady = this.selectionDetector.initialize();

        if (detectorReady) {
            void 0;

            // Listen for selection changes to update scope selector
            this.selectionDetector.on('selection-changed', (count) => {
                this.onSelectionChanged(count);
            });
        } else {
            void 0;
        }

        void 0;
    }

    /**
     * Handle selection count changes
     * @param {number} count Number of selected emails
     */
    onSelectionChanged(count) {
        void 0;

        // Update scope selector if it's currently visible
        if (this.scopeSelectorVisible) {
            this.updateSelectionOption(count);
        }
    }

    /**
     * Show Gmail batch processing scope selector
     * Displays as modal overlay to preserve status elements
     */
    showScopeSelector() {
        void 0;

        // Get visible email count for "Messages on this page" option
        const visibleCount = this.gmailDetector.estimateMessageCount();

        // NEW: Get selected message count for batch selection
        const selectedCount = this.selectionDetector ? this.selectionDetector.getSelectionCount() : 0;

        // Get current date for scope options
        const today = new Date();
        const todayStr = today.toISOString().split('T')[0];

        // Calculate 30 days ago for purchases scope
        const thirtyDaysAgo = new Date();
        thirtyDaysAgo.setDate(thirtyDaysAgo.getDate() - 30);
        const thirtyDaysAgoStr = thirtyDaysAgo.toISOString().split('T')[0];

        // CG Tree icon SVG for reuse
        const cgTreeIcon = `<svg class="carbon-scope-tree-icon" width="20" height="20" viewBox="0 0 100 100" xmlns="http://www.w3.org/2000/svg"><path fill="currentColor" d="M57.7,53c-0.1,2-1.4,4-1.4,4c0-0.7-0.7-3-0.7-3c-0.8,0.3-2.3,0.3-2.3,0.3s0.5,1.3,0.5,2.9c0,1.6-2.1,7.9-2.1,7.9c-2.9-5.1-2.4-9.5-2.4-9.5c-4.4,0.1-8.1-1.1-8.1-1.1l0.9,3.4c-4.2-1.6-4.3-4.2-4.3-4.2l-2.2-0.6c0.4,1.9,3.5,5.5,4.8,6.6c1.4,1.1,4.2,4,5.9,6.1c1.7,2.1,1.9,3.6,2.1,5.2c0.2,1.6-0.8,7.4-0.8,8.4c0,1-1.5,9.4-2.2,10.7c-0.7,1.3-3.5,2.6-3.5,2.6s2.1,0.7,4.3,0.7c2.2,0,7.6-1.1,9-1.1c1.4,0,2,0.4,3.7,0.4c1.7,0,2.1-0.4,2.1-0.4c-4.7-1.7-4.7-2.3-4.7-2.3c-1.5-2.7-2.2-9.9-1.3-19.1c0.9-9.2,7.7-14.8,7.7-14.8C59.3,54.9,57.7,53,57.7,53z M86.8,33.3c4.5-20.8-13.9-19-13.9-19C65.6,1.9,46.1,6.9,46.1,6.9c-21.1-9.3-22.8,7.1-22.8,7.1c-17.2-1.3-9.6,16.3-9.6,16.3c-9.5,16.1,4.6,15.6,4.6,15.6c1.9,12.7,16.6,5.2,16.6,5.2C48.4,56.3,58.5,50.9,58.5,50.9c13.7,11.4,20.2,0,20.2,0C94.4,51.6,86.8,33.3,86.8,33.3z"/></svg>`;
        const cgTreeIconOrange = `<svg class="carbon-scope-option-icon" width="16" height="16" viewBox="0 0 100 100" xmlns="http://www.w3.org/2000/svg"><path fill="#FF6B00" d="M57.7,53c-0.1,2-1.4,4-1.4,4c0-0.7-0.7-3-0.7-3c-0.8,0.3-2.3,0.3-2.3,0.3s0.5,1.3,0.5,2.9c0,1.6-2.1,7.9-2.1,7.9c-2.9-5.1-2.4-9.5-2.4-9.5c-4.4,0.1-8.1-1.1-8.1-1.1l0.9,3.4c-4.2-1.6-4.3-4.2-4.3-4.2l-2.2-0.6c0.4,1.9,3.5,5.5,4.8,6.6c1.4,1.1,4.2,4,5.9,6.1c1.7,2.1,1.9,3.6,2.1,5.2c0.2,1.6-0.8,7.4-0.8,8.4c0,1-1.5,9.4-2.2,10.7c-0.7,1.3-3.5,2.6-3.5,2.6s2.1,0.7,4.3,0.7c2.2,0,7.6-1.1,9-1.1c1.4,0,2,0.4,3.7,0.4c1.7,0,2.1-0.4,2.1-0.4c-4.7-1.7-4.7-2.3-4.7-2.3c-1.5-2.7-2.2-9.9-1.3-19.1c0.9-9.2,7.7-14.8,7.7-14.8C59.3,54.9,57.7,53,57.7,53z M86.8,33.3c4.5-20.8-13.9-19-13.9-19C65.6,1.9,46.1,6.9,46.1,6.9c-21.1-9.3-22.8,7.1-22.8,7.1c-17.2-1.3-9.6,16.3-9.6,16.3c-9.5,16.1,4.6,15.6,4.6,15.6c1.9,12.7,16.6,5.2,16.6,5.2C48.4,56.3,58.5,50.9,58.5,50.9c13.7,11.4,20.2,0,20.2,0C94.4,51.6,86.8,33.3,86.8,33.3z"/></svg>`;
        const cgTreeIconGreen = `<svg class="carbon-scope-option-icon" width="16" height="16" viewBox="0 0 100 100" xmlns="http://www.w3.org/2000/svg"><path fill="#067F38" d="M57.7,53c-0.1,2-1.4,4-1.4,4c0-0.7-0.7-3-0.7-3c-0.8,0.3-2.3,0.3-2.3,0.3s0.5,1.3,0.5,2.9c0,1.6-2.1,7.9-2.1,7.9c-2.9-5.1-2.4-9.5-2.4-9.5c-4.4,0.1-8.1-1.1-8.1-1.1l0.9,3.4c-4.2-1.6-4.3-4.2-4.3-4.2l-2.2-0.6c0.4,1.9,3.5,5.5,4.8,6.6c1.4,1.1,4.2,4,5.9,6.1c1.7,2.1,1.9,3.6,2.1,5.2c0.2,1.6-0.8,7.4-0.8,8.4c0,1-1.5,9.4-2.2,10.7c-0.7,1.3-3.5,2.6-3.5,2.6s2.1,0.7,4.3,0.7c2.2,0,7.6-1.1,9-1.1c1.4,0,2,0.4,3.7,0.4c1.7,0,2.1-0.4,2.1-0.4c-4.7-1.7-4.7-2.3-4.7-2.3c-1.5-2.7-2.2-9.9-1.3-19.1c0.9-9.2,7.7-14.8,7.7-14.8C59.3,54.9,57.7,53,57.7,53z M86.8,33.3c4.5-20.8-13.9-19-13.9-19C65.6,1.9,46.1,6.9,46.1,6.9c-21.1-9.3-22.8,7.1-22.8,7.1c-17.2-1.3-9.6,16.3-9.6,16.3c-9.5,16.1,4.6,15.6,4.6,15.6c1.9,12.7,16.6,5.2,16.6,5.2C48.4,56.3,58.5,50.9,58.5,50.9c13.7,11.4,20.2,0,20.2,0C94.4,51.6,86.8,33.3,86.8,33.3z"/></svg>`;

        // Create scope selector HTML as modal overlay with CG brand header
        const scopeHtml = `
            <div id="carbon-gmail-scope-overlay" class="carbon-gmail-scope-overlay">
                <div id="carbon-gmail-scope-selector" class="carbon-gmail-scope-selector">
                    <div class="carbon-scope-header">
                        <div class="carbon-scope-header-content">
                            ${cgTreeIcon}
                            <span class="carbon-scope-title">Process Emails</span>
                        </div>
                        <button id="carbon-scope-close-btn" class="carbon-scope-close-btn" title="Close">&times;</button>
                    </div>
                    <div class="carbon-scope-options">
                        <a href="#" class="carbon-scope-option" data-scope="first">
                            <span class="carbon-scope-label">${cgTreeIconGreen} First message</span>
                            <span class="carbon-scope-desc">Process most recent email</span>
                        </a>
                        <a href="#" class="carbon-scope-option ${selectedCount === 0 ? 'carbon-scope-disabled' : ''}" data-scope="selected" id="carbon-scope-selected">
                            <span class="carbon-scope-label">${cgTreeIconGreen} Messages selected</span>
                            <span class="carbon-scope-desc" id="carbon-scope-selected-desc">${selectedCount} checked</span>
                            <div class="carbon-scope-warning" id="carbon-scope-selected-warning" style="display: none;"></div>
                        </a>
                        <a href="#" class="carbon-scope-option" data-scope="recent_10">
                            <span class="carbon-scope-label">${cgTreeIconGreen} 10 most recent messages</span>
                            <span class="carbon-scope-desc">Last 10 emails in inbox</span>
                        </a>
                        <a href="#" class="carbon-scope-option carbon-scope-purchases" data-scope="purchase_recent" data-category="purchases" data-count="1">
                            <span class="carbon-scope-label">${cgTreeIconOrange} Most recent purchase</span>
                            <span class="carbon-scope-desc">Latest order receipt or shipping email</span>
                        </a>
                        <a href="#" class="carbon-scope-option carbon-scope-purchases" data-scope="purchase_recent_10" data-category="purchases" data-count="10">
                            <span class="carbon-scope-label">${cgTreeIconOrange} 10 most recent purchases</span>
                            <span class="carbon-scope-desc">Last 10 purchase emails</span>
                        </a>
                        <a href="#" class="carbon-scope-option carbon-scope-purchases" data-scope="purchase_30days" data-category="purchases" data-date-after="${thirtyDaysAgoStr}">
                            <span class="carbon-scope-label">${cgTreeIconOrange} All purchases (last 30 days)</span>
                            <span class="carbon-scope-desc">Purchase emails since ${thirtyDaysAgoStr}</span>
                        </a>
                        <a href="#" class="carbon-scope-option" data-scope="page" data-count="${visibleCount}">
                            <span class="carbon-scope-label">${cgTreeIconGreen} Messages on this page</span>
                            <span class="carbon-scope-desc">~${visibleCount} visible emails</span>
                        </a>
                        <a href="#" class="carbon-scope-option" data-scope="today" data-date="${todayStr}">
                            <span class="carbon-scope-label">${cgTreeIconGreen} All messages from today</span>
                            <span class="carbon-scope-desc">${todayStr}</span>
                        </a>
                        <a href="#" class="carbon-scope-option" data-scope="date_picker">
                            <span class="carbon-scope-label">${cgTreeIconGreen} All messages from date...</span>
                            <span class="carbon-scope-desc">Choose custom date range</span>
                        </a>
                    </div>
                    <div class="carbon-scope-footer">
                        ${cgTreeIconGreen}
                        <span>Calculate carbon footprint from purchase emails</span>
                    </div>
                </div>
            </div>
        `;

        // Inject overlay as a sibling before the tile inside the flex column wrapper
        // (.carbon-guru-analyze-button is display:flex; flex-direction:column)
        // This naturally positions the scope selector above the tile
        const floatingTile = document.querySelector('.carbon-tile');
        if (!floatingTile) {
            console.error('CarbonGuru Gmail: Floating tile not found');
            return;
        }

        // Remove any existing overlay
        this.clearScopeSelector();

        // Collapse tile content so only the button + scope selector are visible
        if (this.contentScript) {
            this.contentScript.collapseTile();
        }

        // Insert before the tile (as a sibling in the flex column, not a child)
        floatingTile.insertAdjacentHTML('beforebegin', DOMPurify.sanitize(scopeHtml, {
            ADD_ATTR: ['data-scope', 'data-count', 'data-date', 'data-category', 'data-date-after']
        }));

        // Attach click handlers to scope options
        this.attachScopeOptionHandlers();

        this.scopeSelectorVisible = true;
    }

    /**
     * Clear scope selector overlay
     */
    clearScopeSelector() {
        const overlay = document.getElementById('carbon-gmail-scope-overlay');
        if (overlay) {
            overlay.remove();
            void 0;
        }
        this.scopeSelectorVisible = false;
    }

    /**
     * Update the "Selected messages" option based on current selection count
     * MVP: Show warning when >5 emails, block when >10 emails
     * @param {number} count Number of selected emails
     */
    updateSelectionOption(count) {
        const option = document.getElementById('carbon-scope-selected');
        const desc = document.getElementById('carbon-scope-selected-desc');
        const warning = document.getElementById('carbon-scope-selected-warning');

        if (!option || !desc || !warning) {
            return;
        }

        // Update count display
        desc.textContent = `${count} checked`;

        // MVP: Hard limit at 10 emails
        if (count > 10) {
            option.classList.add('carbon-scope-disabled', 'carbon-scope-error');
            warning.textContent = `Maximum 10 emails per batch. Uncheck ${count - 10} email(s).`;
            warning.className = 'carbon-scope-warning carbon-scope-warning-error carbon-scope-warning-flash';
            warning.style.display = 'block';
        }
        // MVP: Warning at >5 emails
        else if (count > 5) {
            option.classList.remove('carbon-scope-disabled', 'carbon-scope-error');
            const estimatedTime = Math.ceil(count * 2.5); // ~2.5s per email
            warning.textContent = `Processing ${count} emails may take ${estimatedTime}-${estimatedTime + 10} seconds...`;
            warning.className = 'carbon-scope-warning carbon-scope-warning-info carbon-scope-warning-flash';
            warning.style.display = 'block';
        }
        // No warning
        else if (count > 0) {
            option.classList.remove('carbon-scope-disabled', 'carbon-scope-error');
            warning.style.display = 'none';
        }
        // No selection
        else {
            option.classList.add('carbon-scope-disabled');
            option.classList.remove('carbon-scope-error');
            warning.style.display = 'none';
        }
    }

    /**
     * Attach click handlers to scope selector options
     */
    attachScopeOptionHandlers() {
        // Query within overlay to ensure we find the injected elements
        const overlay = document.getElementById('carbon-gmail-scope-overlay');
        if (!overlay) {
            console.error('CarbonGuru Gmail: Cannot attach scope handlers - overlay not found');
            return;
        }

        // Attach close button handler
        const closeBtn = document.getElementById('carbon-scope-close-btn');
        if (closeBtn) {
            closeBtn.addEventListener('click', (e) => {
                e.preventDefault();
                void 0;
                this.clearScopeSelector();
                // Re-expand tile content that was collapsed when scope selector opened
                if (this.contentScript) {
                    this.contentScript.expandTileWithoutAuthCheck();
                }
                // Reset tile to general open state (cancel the "Get Carbon Footprint" action)
                this.batchProcessingActive = false;
            });
        }

        const scopeOptions = overlay.querySelectorAll('.carbon-scope-option');
        void 0;

        scopeOptions.forEach(option => {
            option.addEventListener('click', async (e) => {
                e.preventDefault();

                // Add active styling
                scopeOptions.forEach(opt => opt.classList.remove('carbon-scope-active'));
                option.classList.add('carbon-scope-active');

                const scope = option.getAttribute('data-scope');
                const count = option.getAttribute('data-count');
                const date = option.getAttribute('data-date');
                const category = option.getAttribute('data-category');
                const dateAfter = option.getAttribute('data-date-after');

                void 0;

                // Handle scope selection
                await this.handleScopeSelection(scope, { count, date, category, dateAfter });
            });

            // Add hover effect
            option.addEventListener('mouseenter', () => {
                option.classList.add('carbon-scope-hover');
            });

            option.addEventListener('mouseleave', () => {
                option.classList.remove('carbon-scope-hover');
            });
        });
    }

    /**
     * Handle scope selection and start processing
     */
    async handleScopeSelection(scope, options = {}) {
        void 0;

        // Clear scope selector overlay immediately
        this.clearScopeSelector();

        // Re-expand tile content so status messages and results are visible
        if (this.contentScript) {
            this.contentScript.expandTileWithoutAuthCheck();
        }

        // Show initial processing status
        this.contentScript.showStatus('Preparing to process emails...', { type: 'info' });

        try {
            let result = null;

            if (scope === 'first') {
                // UI-BASED: Process the currently displayed message
                result = await this.processCurrentMessage();

            } else if (scope === 'selected') {
                // NEW: BATCH SELECTION - Process checked emails sequentially (MVP)
                result = await this.processSelectedMessages();

            } else if (scope === 'recent_10') {
                // DATE-BASED: Search mailbox for 10 most recent purchase emails
                result = await this.processRecentMessages(10);

            } else if (scope === 'purchase_recent') {
                // CATEGORY-BASED: Most recent purchase email (limit: 1)
                result = await this.processCategoryMessages(options.category, parseInt(options.count), 30);

            } else if (scope === 'purchase_recent_10') {
                // CATEGORY-BASED: 10 most recent purchase emails (limit: 10)
                result = await this.processCategoryMessages(options.category, parseInt(options.count), 30);

            } else if (scope === 'purchase_30days') {
                // CATEGORY-BASED: All purchases from last 30 days
                result = await this.processCategoryMessages(options.category, 100, 30);

            } else if (scope === 'page') {
                // UI-BASED: Process messages on current page (same as 'first' for popout)
                result = await this.processCurrentMessage();

            } else if (scope === 'today') {
                // DATE-BASED: Search mailbox for all messages from today
                result = await this.processMessagesFromDate(0);

            } else if (scope === 'date_picker') {
                // DATE-BASED: Show date picker for custom range
                await this.showDatePicker();
                return; // Date picker will handle processing
            }

            // Show result summary
            if (result) {
                this.showBatchResult(result);
            }

        } catch (error) {
            console.error('CarbonGuru Gmail: Scope processing error:', error);
            this.contentScript.showStatus(error.message, { type: 'error', duration: 5000 });
        }
    }

    /**
     * UI-BASED: Process the currently displayed message
     * Extracts message ID from URL and sends to background.js for unified processing
     */
    async processCurrentMessage() {
        void 0;

        // Extract message ID from current page URL
        const messageId = this.gmailDetector.getCurrentMessageId();

        if (!messageId) {
            throw new Error('Cannot determine which email to process. Please open a specific email.');
        }

        void 0;
        this.contentScript.showStatus('Processing this email...', { type: 'info' });

        // Send to background.js for unified async processing (same as cart/product)
        return new Promise((resolve, reject) => {
            const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
            runtime.sendMessage({
                action: 'processGmailEmail',
                messageId: messageId
            }, (response) => {
                if (response && response.success) {
                    void 0;
                    // Background will send analysisStatus messages via polling
                    // Content script will receive them automatically
                    resolve({
                        success: true,
                        process_id: response.process_id
                    });
                } else {
                    const error = response?.error || 'Failed to start email processing';
                    console.error('CarbonGuru Gmail: Processing failed:', error);
                    reject(new Error(error));
                }
            });
        });
    }

    /**
     * NEW: BATCH SELECTION - Process checked/selected emails sequentially (MVP)
     * Calls existing email processing endpoint once per email (no batch endpoint needed)
     */
    async processSelectedMessages() {
        void 0;

        if (!this.selectionDetector) {
            throw new Error('Selection detector not initialized');
        }

        const messageIds = this.selectionDetector.getMessageIdsFromSelection();
        const count = messageIds.length;

        void 0;

        // Validate selection count
        if (count === 0) {
            this.contentScript.showStatus('Please select at least one email to analyze.', { type: 'error' });
            return { processed: 0, failed: 0, skipped: count };
        }

        // MVP: Enforce 10 email limit
        if (count > 10) {
            this.contentScript.showStatus(`Maximum 10 emails per batch. You selected ${count}. Please uncheck ${count - 10} email(s).`, { type: 'error' });
            return { processed: 0, failed: 0, skipped: count };
        }

        // Show initial status
        this.contentScript.showStatus(`Processing ${count} selected email${count > 1 ? 's' : ''}...`, { type: 'info' });

        const results = {
            processed: 0,
            failed: 0,
            skipped: 0,
            totalCarbon: 0,
            totalCost: 0,
            errors: []
        };

        // FIX: Route through background.js for unified status polling
        // Process emails sequentially (like cart processing - one at a time)
        // Background will poll and send analysisStatus messages automatically
        const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;

        for (let i = 0; i < messageIds.length; i++) {
            const messageId = messageIds[i];

            try {
                void 0;

                // Send to background.js for unified async processing
                const result = await new Promise((resolve, reject) => {
                    runtime.sendMessage({
                        action: 'processGmailEmail',
                        messageId: messageId
                    }, (response) => {
                        if (response && response.success) {
                            void 0;
                            resolve(response);
                        } else {
                            const error = response?.error || 'Failed to start email processing';
                            console.error(`[GmailBatchSelection] Email ${i + 1}/${count} failed:`, error);
                            reject(new Error(error));
                        }
                    });
                });

                if (result && result.success) {
                    results.processed++;
                } else {
                    results.failed++;
                    results.errors.push({
                        messageId: messageId,
                        error: result ? result.error : 'Unknown error'
                    });
                }

            } catch (error) {
                console.error(`[GmailBatchSelection] Failed to process ${messageId}:`, error);
                results.failed++;
                results.errors.push({
                    messageId: messageId,
                    error: error.message
                });
            }
        }

        // Log final results (rely on existing extension notification system for MVP)
        void 0;

        if (results.failed > 0) {
            void 0;
        }

        return results;
    }

    /**
     * DATE-BASED: Search mailbox for N most recent purchase-related messages
     * Searches last 30 days for emails matching purchase keywords, returns most recent N
     */
    async processRecentMessages(count) {
        void 0;

        this.contentScript.showStatus(`Searching for ${count} most recent purchase emails...`, { type: 'info' });

        // Send to background.js for unified async processing (same as cart/product)
        // Background.js will poll and send status messages automatically
        return new Promise((resolve, reject) => {
            const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
            runtime.sendMessage({
                action: 'processGmailDateRange',
                options: {
                    days_back: 30,
                    limit: count
                }
            }, (response) => {
                if (response && response.success) {
                    void 0;
                    // Background will send analysisStatus messages via polling
                    // Content script will receive them automatically
                    resolve({
                        success: true,
                        process_id: response.process_id
                    });
                } else {
                    const error = response?.error || 'Failed to start email processing';
                    console.error('CarbonGuru Gmail: Processing failed:', error);
                    reject(new Error(error));
                }
            });
        });
    }

    /**
     * Process batch of messages by IDs
     */
    async processBatchMessages(messageIds) {
        void 0;

        if (messageIds.length === 0) {
            throw new Error('No messages found on this page');
        }

        // Show confirmation
        this.contentScript.showStatus(`Processing ${messageIds.length} emails from this page...`, { type: 'info' });

        this.batchProcessingActive = true;

        const result = await this.gmailIntegration.processBatchEmails(
            messageIds,
            (status) => this.handleGmailStatusCallback(status)
        );

        this.batchProcessingActive = false;

        return result;
    }

    /**
     * DATE-BASED: Process messages from date range (days back)
     * Searches Gmail API for all purchase-related messages within the date range
     */
    async processMessagesFromDate(daysBack) {
        void 0;

        const dateStr = daysBack === 0 ? 'today' : `last ${daysBack} days`;
        this.contentScript.showStatus(`Processing emails from ${dateStr}...`, { type: 'info' });

        this.batchProcessingActive = true;

        // Send to background.js for unified async processing (same as cart/product)
        const result = await new Promise((resolve, reject) => {
            const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
            runtime.sendMessage({
                action: 'processGmailDateRange',
                options: {
                    days_back: daysBack
                }
            }, (response) => {
                if (response && response.success) {
                    void 0;
                    resolve({
                        success: true,
                        process_id: response.process_id
                    });
                } else {
                    const error = response?.error || 'Failed to start email processing';
                    console.error('CarbonGuru Gmail: Processing failed:', error);
                    reject(new Error(error));
                }
            });
        });

        this.batchProcessingActive = false;

        return result;
    }

    /**
     * CATEGORY-BASED: Process messages from Gmail category (e.g., Purchases)
     * Uses Gmail API query string for accurate category filtering
     */
    async processCategoryMessages(category, limit, daysBack) {
        void 0;

        const categoryDisplay = category.charAt(0).toUpperCase() + category.slice(1);
        this.contentScript.showStatus(`Processing ${categoryDisplay} emails...`, { type: 'info' });

        this.batchProcessingActive = true;

        // Send to background.js for unified async processing (same as cart/product)
        void 0;
        const result = await new Promise((resolve, reject) => {
            const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;

            void 0;
            void 0;

            runtime.sendMessage({
                action: 'processGmailDateRange',
                options: {
                    category: category,
                    limit: limit,
                    days_back: daysBack
                }
            }, (response) => {
                void 0;

                // Check for runtime errors
                const lastError = runtime.lastError;
                if (lastError) {
                    console.error('CarbonGuru Gmail: Runtime error:', lastError);
                    reject(new Error(lastError.message || 'Runtime error occurred'));
                    return;
                }

                if (response && response.success) {
                    void 0;
                    resolve({
                        success: true,
                        process_id: response.process_id
                    });
                } else {
                    const error = response?.error || 'Failed to start email processing';
                    console.error('CarbonGuru Gmail: Processing failed:', error);
                    reject(new Error(error));
                }
            });
        });

        this.batchProcessingActive = false;

        return result;
    }

    /**
     * Show date picker for custom date range
     */
    async showDatePicker() {
        void 0;

        // Clear any existing overlay
        this.clearScopeSelector();

        // Create date picker HTML as modal overlay
        const datePickerHtml = `
            <div id="carbon-gmail-scope-overlay" class="carbon-gmail-scope-overlay">
                <div id="carbon-gmail-date-picker" class="carbon-gmail-date-picker">
                    <div class="carbon-date-picker-header">Process emails from:</div>
                    <div class="carbon-date-picker-input">
                        <label for="carbon-date-input">Days back:</label>
                        <input type="number" id="carbon-date-input" min="1" max="365" value="7" />
                        <span class="carbon-date-picker-hint">(1-365 days)</span>
                    </div>
                    <div class="carbon-date-picker-buttons">
                        <button id="carbon-date-picker-confirm" class="carbon-btn carbon-btn-primary">Process Emails</button>
                        <button id="carbon-date-picker-cancel" class="carbon-btn carbon-btn-secondary">Cancel</button>
                    </div>
                </div>
            </div>
        `;

        // Inject overlay as sibling before tile (same as scope selector)
        const floatingTile = document.querySelector('.carbon-tile');
        if (!floatingTile) return;

        floatingTile.insertAdjacentHTML('beforebegin', DOMPurify.sanitize(datePickerHtml));
        this.scopeSelectorVisible = true;

        // Attach handlers
        const confirmBtn = document.getElementById('carbon-date-picker-confirm');
        const cancelBtn = document.getElementById('carbon-date-picker-cancel');
        const dateInput = document.getElementById('carbon-date-input');

        confirmBtn.addEventListener('click', async () => {
            const daysBack = parseInt(dateInput.value, 10);

            if (isNaN(daysBack) || daysBack < 1 || daysBack > 365) {
                this.contentScript.showStatus('Please enter a valid number of days (1-365)', {
                    type: 'warning',
                    duration: 3000
                });
                return;
            }

            // Clear overlay before processing
            this.clearScopeSelector();

            // Show initial status
            this.contentScript.showStatus('Preparing to process emails...', { type: 'info' });

            // Process messages from custom date
            const result = await this.processMessagesFromDate(daysBack);
            if (result) {
                this.showBatchResult(result);
            }
        });

        cancelBtn.addEventListener('click', () => {
            // Return to scope selector
            this.showScopeSelector();
        });
    }

    /**
     * Handle Gmail status callbacks from integration
     * DEPRECATED: No longer used - background.js now sends standard analysisStatus messages
     * Content script receives them via the normal message listener
     */
    handleGmailStatusCallback(status) {
        void 0;
        // This method is no longer used - background.js sends analysisStatus messages directly
        // Content script's setupMessageListener() handles them automatically
    }

    /**
     * Show batch processing result summary
     * DEPRECATED: Result now comes via standard analysisComplete message from background.js
     */
    showBatchResult(result) {
        void 0;
        // Results now come via analysisComplete messages from background.js
        // Content script's normal message handling will display them using showCartCompletionStatus()
    }

    /**
     * Check if scope selector is visible
     */
    isScopeSelectorVisible() {
        return this.scopeSelectorVisible;
    }

    /**
     * Check if batch processing is active
     */
    isBatchProcessing() {
        return this.batchProcessingActive;
    }
}

// Export for use in content script
if (typeof module !== 'undefined' && module.exports) {
    module.exports = FloatingTileGmailExtension;
}
